<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Requestforquote
 * @author    Webkul Software Private Limited
 * @copyright Copyright (c) Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\Requestforquote\Block\Account\Customer;

class QuoteList extends \Magento\Framework\View\Element\Template
{
    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    private $customerFactory;

    /**
     * @var \\Webkul\Requestforquote\Model\ResourceModel\Quote\CollectionFactory
     */
    private $quoteFactory;

    /**
     * @var \Webkul\Requestforquote\Model\ResourceModel\Info\CollectionFactory
     */
    private $infoFactory;

    /**
     * @var \Webkul\Requestforquote\Model\QuoteFactory
     */
    private $quote;
    
    /**
     * @var \Webkul\Marketplace\Helper\Data
     */
    private $mpHelper;
    
    /**
     * __construct function
     *
     * @param \Magento\Framework\View\Element\Template\Context                    $context
     * @param \Magento\Customer\Model\CustomerFactory                             $customerFactory
     * @param \Webkul\Requestforquote\Model\ResourceModel\Quote\CollectionFactory $quoteFactory
     * @param \Webkul\Requestforquote\Model\ResourceModel\Info\CollectionFactory  $infoFactory
     * @param \Webkul\Requestforquote\Model\QuoteFactory                          $quote
     * @param \Webkul\Marketplace\Helper\Data                                     $mpHelper
     * @param array                                                               $data
     */
    public function __construct(
        \Magento\Framework\View\Element\Template\Context $context,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Webkul\Requestforquote\Model\ResourceModel\Quote\CollectionFactory $quoteFactory,
        \Webkul\Requestforquote\Model\ResourceModel\Info\CollectionFactory $infoFactory,
        \Webkul\Requestforquote\Model\QuoteFactory $quote,
        \Webkul\Marketplace\Helper\Data $mpHelper,
        \Webkul\Requestforquote\Helper\Data $helper,
        array $data = []
    ) {
        $this->customerFactory = $customerFactory;
        $this->quoteFactory = $quoteFactory;
        $this->infoFactory = $infoFactory;
        $this->quote = $quote;
        $this->mpHelper = $mpHelper;
        $this->helper = $helper;
        parent::__construct($context, $data);

        $customerId = $this->mpHelper->getCustomerId();

        $quoteCollection = $this->quoteFactory->create();
        $quoteId = $this->getRequest()->getParam('id');
        $filterStatus = 1;
        if ($this->getRequest()->getParam('status') != '') {
            $status = $this->getRequest()->getParam('status');
            $filterStatus = $status != '' ? $status : '';
        }
        if ($filterStatus == 0) {
            $quoteCollection->addFieldToFilter('entity_id', $quoteId);
            if ($filterStatus!='') {
                $quoteCollection->addFieldToFilter('status', $filterStatus);
            }
            $this->setCollection($quoteCollection);
        } else {
            $quotes = $this->infoFactory->create()
                ->addFieldToFilter('quote_id', $quoteId);
            if ($filterStatus!='') {
                $quotes->addFieldToFilter('customer_status', $filterStatus);
            }
            $this->setCollection($quotes);
        }
    }

    /**
     * @return array
     */
    public function requestData()
    {
        return $this->getRequest()->getParams();
    }

    /**
     * getRequestedQuote get info collection of quotes
     *
     * @return \Webkul\Requestforquote\Model\ResourceModel\Info\Collection
     */
    public function getRequestedQuote()
    {
        return $this->getCollection();
    }
    
    /**
     * _prepareLayout
     *
     * @return void
     */
    protected function _prepareLayout()
    {
        parent::_prepareLayout();
        if ($this->getCollection()) {
            $pager = $this->getLayout()->createBlock(
                \Magento\Theme\Block\Html\Pager::class,
                'taxmanager.taxrules.list.pager'
            )->setCollection(
                $this->getCollection()
            );
            $this->setChild('pager', $pager);
            $this->getCollection()->load();
        }

        return $this;
    }

    /**
     * @return string
     */
    public function getPagerHtml()
    {
        return $this->getChildHtml('pager');
    }

    /**
     * Get requested quote count by status
     *
     * @param int $quoteId
     * @param int $status
     *
     * @return Webkul_Requestforquote_Model_Quote
     */
    public function getRequestedQuoteCountByStatus($quoteId, $status)
    {
        if ($status == 0) {
            $quotes = $this->quoteFactory->create()
                ->addFieldToFilter('entity_id', $quoteId)
                ->addFieldToFilter('status', $status);
            return count($quotes);
        }
        $quotes = $this->infoFactory->create()
            ->addFieldToFilter('quote_id', $quoteId)
            ->addFieldToFilter('customer_status', $status);
        return count($quotes);
    }

    /**
     * Get requested quote data
     *
     * @return Webkul_Requestforquote_Model_Quote
     */
    public function getQuoteById()
    {
        $quoteId = $this->getRequest()->getParam('id');
        return $this->quote->create()->load($quoteId);
    }

    /**
     * Get requested quote view url by quoteId
     *
     * @param  $quoteId
     * @return string
     */
    public function getViewUrl($quoteId, $isPending = false)
    {
        if ($isPending) {
            return $this->getUrl('*/*/quoteview', ['id' => $quoteId]);
        }
        return $this->getUrl('*/*/sellerquoteview', ['id' => $quoteId]);
    }

    /**
     * Get requested open quote url
     *
     * @return string
     */
    public function getOpenQuoteUrl()
    {
        $quoteId = $this->getRequest()->getParam('id');
        return $this->getUrl('*/*/quotelist', ['id' => $quoteId, 'status' => 1]);
    }

    /**
     * Get requested pending quote url
     *
     * @return string
     */
    public function getPendingQuoteUrl()
    {
        $quoteId = $this->getRequest()->getParam('id');
        return $this->getUrl('*/*/quotelist', ['id' => $quoteId, 'status' => 0]);
    }

    /**
     * Get requested answered quote url
     *
     * @return string
     */
    public function getAnsweredQuoteUrl()
    {
        $quoteId = $this->getRequest()->getParam('id');
        return $this->getUrl('*/*/quotelist', ['id' => $quoteId, 'status' => 2]);
    }

    /**
     * Get requested solved quote url
     *
     * @return string
     */
    public function getSolvedQuoteUrl()
    {
        $quoteId = $this->getRequest()->getParam('id');
        return $this->getUrl('*/*/quotelist', ['id' => $quoteId, 'status' => 3]);
    }

    /**
     * Get requested closed quote url
     *
     * @return string
     */
    public function getClosedQuoteUrl()
    {
        $quoteId = $this->getRequest()->getParam('id');
        return $this->getUrl('*/*/quotelist', ['id' => $quoteId, 'status' => 4]);
    }

    /**
     * Get seller name by sellerId
     *
     * @param  $sellerId
     * @return string
     */
    public function getSellerName($sellerId)
    {
        $customer = $this->customerFactory->create()->load($sellerId);
        return $customer['firstname']?$customer['firstname']." ".$customer['lastname']:__('Admin');
    }

    public function getHelperObj()
    {
        return $this->helper;
    }

    public function getRequestObj()
    {
        return $this->getRequest();
    }
}
