<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Requestforquote
 * @author    Webkul Software Private Limited
 * @copyright Copyright (c) Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\Requestforquote\Block\Account\Customer;

use Magento\Framework\Filesystem\Io\File;

class QuoteView extends \Magento\Framework\View\Element\Template
{

    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    private $customerFactory;
    
    /**
     * @var \Magento\Catalog\Model\ProductFactory
     */
    private $catalogProductFactory;
    
    /**
     * @var \Webkul\Requestforquote\Model\QuoteFactory
     */
    private $quote;
    
    /**
     * @var \Webkul\Requestforquote\Model\InfoFactory
     */
    private $info;
    
    /**
     * @var \Webkul\Requestforquote\Model\ResourceModel\Info\CollectionFactory
     */
    private $infoFactory;
    
    /**
     * @var \Webkul\Marketplace\Model\ProductFactory
     */
    private $product;
    
    /**
     * @var \Webkul\Requestforquote\Helper\Data
     */
    private $helper;
    
    /**
     * @var \Webkul\Marketplace\Helper\Data
     */
    private $mpHelper;

    /**
     * @var \Webkul\Marketplace\Model\ResourceModel\Seller\Collection
     */
    private $mpSellerColl;

    /**
     * __construct
     *
     * @param \Magento\Framework\View\Element\Template\Context                           $context
     * @param \Magento\Customer\Model\CustomerFactory                                    $customerFactory
     * @param \Magento\Catalog\Model\ProductFactory                                      $catalogProductFactory
     * @param \Webkul\Requestforquote\Model\QuoteFactory                                 $quote
     * @param \Webkul\Requestforquote\Model\InfoFactory                                  $info
     * @param \Webkul\Requestforquote\Model\ResourceModel\Info\CollectionFactory         $infoFactory
     * @param \Webkul\Marketplace\Model\ProductFactory                                   $product
     * @param \Webkul\Requestforquote\Helper\Data                                        $helper
     * @param \Webkul\Marketplace\Helper\Data                                            $mpHelper
     * @param \Webkul\Marketplace\Model\ResourceModel\Seller\CollectionFactory           $mpSellerColl
     * @param array                                                                      $data
     */
    public function __construct(
        \Magento\Framework\View\Element\Template\Context $context,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Magento\Catalog\Model\ProductFactory $catalogProductFactory,
        \Webkul\Requestforquote\Model\QuoteFactory $quote,
        \Webkul\Requestforquote\Model\InfoFactory $info,
        \Webkul\Requestforquote\Model\ResourceModel\Info\CollectionFactory $infoFactory,
        \Webkul\Marketplace\Model\ProductFactory $product,
        \Webkul\Requestforquote\Helper\Data $helper,
        \Magento\Catalog\Helper\Image $catalogImgHelper,
        \Magento\Catalog\Helper\Output $catalogOpHelper,
        \Magento\Framework\Pricing\Helper\Data $priceHelper,
        \Webkul\Marketplace\Helper\Data $mpHelper,
        \Webkul\Marketplace\Model\ResourceModel\Seller\CollectionFactory $mpSellerColl,
        File $file,
        array $data = []
    ) {
        $this->customerFactory = $customerFactory;
        $this->catalogProductFactory = $catalogProductFactory;
        $this->quote = $quote;
        $this->info = $info;
        $this->infoFactory = $infoFactory;
        $this->infoFactory = $infoFactory;
        $this->product = $product;
        $this->helper = $helper;
        $this->catalogImgHelper = $catalogImgHelper;
        $this->mpHelper = $mpHelper;
        $this->catalogOpHelper = $catalogOpHelper;
        $this->priceHelper = $priceHelper;
        $this->mpSellerColl = $mpSellerColl;
        parent::__construct($context, $data);
        $sellerQuoteId = $this->getRequest()->getParam('id');
        $this->file = $file;
    }

    /**
     * get filter data for rate
     *
     * @return array
     */
    public function requestData()
    {
        return $this->getRequest()->getParams();
    }

    public function getApprovedQuotePrice($collection = null)
    {
        if ($collection->getSize() == 1) {
            foreach ($collection as $model) {
                $price = $model->getQuotePrice();
                break;
            }
            return $price;
        }
    }
    
    /**
     * _prepareLayout prepare pager for rules list
     *
     * @return void
     */
    protected function _prepareLayout()
    {
        parent::_prepareLayout();
        if ($this->getCollection()) {
            $pager = $this->getLayout()->createBlock(
                \Magento\Theme\Block\Html\Pager::class,
                'taxmanager.taxrules.list.pager'
            )->setCollection(
                $this->getCollection()
            );
            $this->setChild('pager', $pager);
            $this->getCollection()->load();
        }

        return $this;
    }
    
    /**
     * Get requested quote info collection by quoteId
     *
     * @param  int $quoteId
     * @return Webkul_Requestforquote_Model_ResourceInfo_Collection
     */
    public function getQuoteInfoByQuoteId($quoteId)
    {
        $quotes = $this->infoFactory->create()
            ->addFieldToFilter('quote_id', $quoteId);
        return $quotes;
    }
    
    /**
     * Get requested quote data by id
     *
     * @param  int $quoteId
     * @return Webkul_Requestforquote_ModelQuote
     */
    public function getQuoteById($quoteId)
    {
        return $this->quote->create()->load($quoteId);
    }

    /**
     * getQuoteInfoById load the quote information model
     *
     * @param  [int] $id
     * @return int
     */
    public function getQuoteInfoById($id)
    {
        $quotes = $this->quote->create()->load($id);
        return $quotes;
    }
    
    /**
     * Get product info by productId
     *
     * @param  int $productId
     * @return Magento_Catalog_Model_Product
     */
    public function getProductDetails($productId)
    {
        return $this->catalogProductFactory->create()->load($productId);
    }

    /**
     * Get requested quote by seller lust url by quoteId
     *
     * @param  $quoteId
     * @return string
     */
    public function getViewQuoteBySellerUrl($quoteId)
    {
        return $this->getUrl('*/*/quotelist', ['id' => $quoteId]);
    }

    /**
     * isSellerProduct get the information that the product is seller product or not?
     *
     * @param  [int] $productId
     * @param  [int] $sellerId
     * @return int
     */
    public function isSellerProduct($productId, $sellerId)
    {
        $collection = $this->product->create()->getCollection()
            ->addFieldToFilter('mageproduct_id', $productId)
            ->addFieldToFilter('seller_id', $sellerId);
        return count($collection);
    }

    /**
     * Get seller name
     *
     * @param  $sellerId
     * @return string
     */
    public function getSellerNameById($sellerId)
    {
        $customer = $this->customerFactory->create()->load($sellerId);
        return $customer['firstname']." ".$customer['lastname'];
    }

    /**
     * Get customer name
     *
     * @return string
     */
    public function getCustomerNameById($customerId)
    {
        $customer = $this->customerFactory->create()->load($customerId);
        return $customer['firstname']." ".$customer['lastname'];
    }

    /**
     * get the logo of seller
     *
     * @return string
     */
    public function getSellerLogoBySellerId()
    {
        $sellerId = $this->mpHelper->getCustomerId();
        $logo = 'noimage.png';
        $userdata = $this->mpSellerColl->create()
            ->addFieldToFilter('seller_id', $sellerId);
        if ($userdata->getSize()) {
            foreach ($userdata as $key => $value) {
                if ($value->getLogopic()) {
                    $logo = $value->getLogopic();
                }
            }
        }
        return $logo;
    }

    /**
     * get the country name from country code
     *
     * @param  string $countryCode
     * @return string
     */
    public function getCountryNameByCode($countryCode = "")
    {
        return $this->helper->getCountryname($countryCode);
    }

    public function getHelperObj()
    {
        return $this->helper;
    }

    public function getCatalogImgHelper()
    {
        return $this->catalogImgHelper;
    }

    public function getCatalogOpHelper()
    {
        return $this->catalogOpHelper;
    }
    
    public function getPriceHelper()
    {
        return $this->priceHelper;
    }

    public function getRequestObj()
    {
        return $this->getRequest();
    }

    public function getPathInfo($sampleImage)
    {
        return $this->file->getPathInfo($sampleImage);
    }
}
