<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Requestforquote
 * @author    Webkul Software Private Limited
 * @copyright Copyright (c) Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\Requestforquote\Controller\Account;

use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\View\Result\PageFactory;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Exception\LocalizedException;

/**
 * Webkul Requestforquote Landing page UpdateRequestforquote Controller.
 */
class Post extends Action
{
    /**
     * @var Magento\Framework\View\Result\PageFactory
     */
    private $resultPageFactory;

    /**
     * @var \Webkul\Requestforquote\Model\QuoteFactory
     */
    private $quoteFactory;

    /**
     * @var \Webkul\Requestforquote\Model\InfoFactory
     */
    private $infoFactory;

    /**
     * @var \Webkul\Requestforquote\Helper\Data
     */
    private $dataHelper;

    /**
     * @var \Webkul\Marketplace\Model\SellerFactory
     */
    private $userProfile;

    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    private $customer;

    /**
     * @var \Magento\Framework\Session\SessionManagerInterface
     */
    private $session;

    /**
     * @var \Webkul\Marketplace\Helper\Data
     */
    private $mpHelper;

    /**
     * __construct
     *
     * @param Context                                            $context
     * @param PageFactory                                        $resultPageFactory
     * @param \Webkul\Requestforquote\Model\QuoteFactory         $quoteFactory
     * @param \Webkul\Requestforquote\Model\InfoFactory          $infoFactory
     * @param \Webkul\Requestforquote\Helper\Data                $dataHelper
     * @param \Webkul\Marketplace\Model\SellerFactory            $userProfileFactory
     * @param \Magento\Customer\Model\CustomerFactory            $customerFactory
     * @param \Magento\Framework\Session\SessionManagerInterface $session
     * @param \Webkul\Marketplace\Helper\Data                    $mpHelper
     */
    public function __construct(
        Context $context,
        PageFactory $resultPageFactory,
        \Webkul\Requestforquote\Model\QuoteFactory $quoteFactory,
        \Webkul\Requestforquote\Model\InfoFactory $infoFactory,
        \Webkul\Requestforquote\Helper\Data $dataHelper,
        \Magento\Catalog\Model\ProductRepository $productRepository,
        \Webkul\Marketplace\Model\SellerFactory $userProfileFactory,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Magento\Framework\Session\SessionManagerInterface $session,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Webkul\Marketplace\Helper\Data $mpHelper
    ) {
        $this->resultPageFactory = $resultPageFactory;
        $this->quoteFactory = $quoteFactory;
        $this->productRepository = $productRepository;
        $this->infoFactory = $infoFactory;
        $this->dataHelper = $dataHelper;
        $this->userProfile = $userProfileFactory;
        $this->customer = $customerFactory;
        $this->session = $session;
        $this->mpHelper = $mpHelper;
        $this->_date = $date;
        parent::__construct($context);
    }

    public function execute()
    {
        $postData = $this->session->getRequestForQuoteFormData();
        $actualProduct = '';
        try {
            if (isset($postData['product_id']) && $postData['product_id'] != null) {
                $actualProduct = $this->productRepository->getById($postData['product_id']);
            }
            if (!$this->dataHelper->checkForSellerExistence()
                && !$this->dataHelper->getApprovalRequiredStatus()
            ) {
                throw new LocalizedException(
                    __('Cannot request for quote. No seller associated with the current site')
                );
            }
            $validateData = $this->dataHelper->validatePostData($postData);
            if ($validateData['status'] == false) {
                throw new LocalizedException(__($validateData['error_msg']));
            }
            if (isset($postData['sample_images'])) {
                $postData['sample_images'] = implode(',', $postData['sample_images']);
            }
            $customerId = $this->mpHelper->getCustomerId();
            $postData['customer_id'] = $customerId;
            $customer = $this->customer->create()->load($customerId);
            if (!$customer->getPrimaryShippingAddress()) {
                $this->session->setRequestForQuoteFormData('');
                $this->session->unsetRequestForQuoteFormData();
                $this->messageManager->addError(__('Please set your default shipping address first.'));
                $this->_redirect('requestforquote/index/index');
                return;
            }
            $postData['destination_country'] = $customer->getPrimaryShippingAddress()->getCountryId();
            if (!$this->dataHelper->getApprovalRequiredStatus()) {
                $postData['status'] = 1;
            } else {
                $postData['status'] = 0;
            }
            if (!$this->dataHelper->checkForSellerExistence() && $this->dataHelper->getApprovalRequiredStatus()) {
                $postData['status'] = 0;
            }
            $model = $this->quoteFactory->create();
            $postData['subject'] = htmlentities($postData['subject']);
            $postData['product_weight'] = ($actualProduct != null)?$actualProduct->getWeight():0;
            $postData['description'] = $postData['description'];
            $postData['tax_class_id'] = $postData['tax_class_id'];
            $postData['created_at'] = $this->_date->gmtDate();
            $postData['updated_at'] = $this->_date->gmtDate();
            $model->setData($postData);
            $quoteId = $model->save()->getId();
            if (!$this->dataHelper->getApprovalRequiredStatus()) {
                $collection = $this->userProfile->create()->getCollection()
                    ->addFieldToFilter('is_seller', ['eq'=>1])
                    ->addFieldToSelect('seller_id')
                    ->distinct(true);
                foreach ($collection as $key => $value) {
                    /* Send Quote Mail To Seller */
                    $seller = $this->customer->create()->load($value['seller_id']);
                    if ($seller->getId()) {
                        /* save quote info for seller */
                        $modelInfo = $this->infoFactory->create();
                        $modelInfo->setData($postData);
                        $modelInfo->setSellerId($value['seller_id']);
                        $modelInfo->setQuoteId($quoteId);
                        $modelInfo->setBulkQuoteQty($postData['bulk_order_qty']);
                        $modelInfo->setStatus(0);
                        $modelInfo->setCustomerStatus(2);
                        $modelInfo->setCreatedAt($this->_date->gmtDate());
                        $modelInfo->setUpdatedAt($this->_date->gmtDate());
                        $modelInfo->save();

                        /* send mail to seller */

                        /* Assign values for your template variables  */
                        $emailTempVariables = [];

                        $sellerName = $seller->getFirstname()." ".$seller->getLastname();
                        $sellerEmail = $seller->getEmail();
                        $receiverInfo = [
                            'name' => $sellerName,
                            'email' => $sellerEmail
                        ];
                        $customerName = $customer->getFirstname()." ".$customer->getLastname();
                        $customerEmail = $customer->getEmail();
                        $senderInfo = [
                            'name' => $customerName,
                            'email' => $customerEmail
                        ];
                        $emailTempVariables['name'] = $sellerName;
                        $emailTempVariables['subject'] = $postData['subject'];
                        $emailTempVariables['quantity'] = $postData['bulk_order_qty'];
                        try {
                            $template = \Webkul\Requestforquote\Helper\Data::XML_PATH_EMAIL_TO_SELLER;
                            $this->dataHelper->customMailSendMethod(
                                $emailTempVariables,
                                $senderInfo,
                                $receiverInfo,
                                $template
                            );
                        } catch (\Exception $e) {
                            $this->messageManager->addError($e->getMessage());
                        }
                    }
                }
            }
            $this->messageManager->addSuccess(__('Request for quote was successfully sent.'));
            $this->session->setRequestForQuoteFormData('');
            $this->session->unsetRequestForQuoteFormData();
            $this->_redirect('requestforquote/index/index');
        } catch (\Exception $e) {
            $this->messageManager->addError($e->getMessage());
            $this->_redirect('requestforquote/index/index');
            return;
        }
    }
}
