<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Requestforquote
 * @author    Webkul Software Private Limited
 * @copyright Copyright (c) Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\Requestforquote\Controller\Adminhtml\Quote;

use Magento\Backend\App\Action\Context;
use Magento\Framework\View\Result\PageFactory;
use Magento\Ui\Component\MassAction\Filter;
use Magento\Framework\Exception\LocalizedException;

class SentQuoteToSeller extends \Magento\Backend\App\Action
{
    /**
     * @var \Magento\Framework\View\Result\PageFactory
     */
    private $resultPageFactory;

    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    private $customerFactory;

    /**
     * @var \Webkul\Requestforquote\Helper\Data
     */
    private $dataHelper;

    /**
     * @var \Webkul\Marketplace\Model\SellerFactory
     */
    private $userProfile;

    /**
     * @var \Webkul\Requestforquote\Model\ResourceModel\Quote\CollectionFactory
     */
    private $quoteCollection;

    /**
     * @var \Webkul\Requestforquote\Model\InfoFactory
     */
    private $infoFactory;

    /**
     * @var Filter
     */
    private $filter;

    /**
     * __construct
     *
     * @param \Magento\Backend\App\Action\Context                                 $context
     * @param \Magento\Customer\Model\CustomerFactory                             $customerFactory
     * @param \Webkul\Requestforquote\Helper\Data                                 $dataHelper
     * @param \Webkul\Requestforquote\Model\ResourceModel\Quote\CollectionFactory $quoteCollection
     * @param \Webkul\Marketplace\Model\SellerFactory                             $userProfileFactory
     * @param \Webkul\Requestforquote\Model\InfoFactory                           $infoFactory
     * @param Filter                                                              $filter
     * @param \Magento\Framework\View\Result\PageFactory                          $resultPageFactory
     */
    public function __construct(
        \Magento\Backend\App\Action\Context $context,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Webkul\Requestforquote\Helper\Data $dataHelper,
        \Webkul\Requestforquote\Model\ResourceModel\Quote\CollectionFactory $quoteCollection,
        \Webkul\Marketplace\Model\SellerFactory $userProfileFactory,
        \Webkul\Requestforquote\Model\InfoFactory $infoFactory,
        Filter $filter,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Framework\View\Result\PageFactory $resultPageFactory
    ) {
        parent::__construct($context);
        $this->customerFactory = $customerFactory;
        $this->dataHelper = $dataHelper;
        $this->resultPageFactory = $resultPageFactory;
        $this->quoteCollection = $quoteCollection;
        $this->userProfile = $userProfileFactory;
        $this->infoFactory = $infoFactory;
        $this->filter = $filter;
        $this->_date = $date;
    }

    public function execute()
    {
        $resultRedirect = $this->resultRedirectFactory->create();
        $quotes = $this->filter->getCollection($this->quoteCollection->create());
        $total = $quotes->addFieldToFilter('status', ['eq'=>0])->getSize();
        try {
            if (!$this->dataHelper->checkForSellerExistence()) {
                throw new LocalizedException(
                    __(
                        'Cannot send quote to seller(s). No seller(s) are associated with the current site'
                    )
                );
            }
            foreach ($quotes as $quote) {
                if ($quote->getStatus() == 0) {
                    $quote->setStatus(1)->save();
                    $postData = $quote->getData();
                    $customerId = $quote->getCustomerId();
                    $customer = $this->customerFactory->create()->load($customerId);
                    $collection = $this->userProfile->create()->getCollection()
                        ->addFieldToFilter('is_seller', ['eq'=>1])
                        ->addFieldToSelect('seller_id')
                        ->distinct(true);
                    foreach ($collection as $key => $value) {
                        /* Send Quote Mail To Seller */
                        $this->sendMail($value, $quote, $postData, $customer);
                    }
                }
            }
            $this->messageManager->addSuccess(
                __('Total of %1 record(s) were successfully sent to seller(s)', $total)
            );
        } catch (\Exception $e) {
            $this->messageManager->addError($e->getMessage());
        }
        $resultRedirect->setPath('requestforquote/quote/index');
        return $resultRedirect;
    }

    /*
     * Check permission via ACL resource
     */
    protected function _isAllowed()
    {
        return $this->_authorization->isAllowed('Webkul_Requestforquote::quote_index');
    }

    public function sendMail($value, $quote, $postData, $customer)
    {
        $seller = $this->customerFactory->create()->load($value['seller_id']);
        if ($seller->getId()) {
            /* save quote info for seller */
            $modelInfo = $this->infoFactory->create();
            $modelInfo->setSellerId($value['seller_id']);
            $modelInfo->setQuoteId($quote->getEntityId());
            $modelInfo->setBulkQuoteQty($postData['bulk_order_qty']);
            $modelInfo->setStatus(0);
            $modelInfo->setCustomerStatus(2);
            $modelInfo->setCreatedAt($this->_date->gmtDate());
            $modelInfo->setUpdatedAt($this->_date->gmtDate());
            $modelInfo->save();

            /* send mail to seller */
            $emailTempVariables = [];

            $sellerName = $seller->getFirstname()." ".$seller->getLastname();
            $sellerEmail = $seller->getEmail();
            $receiverInfo = [
                'name' => $sellerName,
                'email' => $sellerEmail
            ];
            $customerName = $customer->getFirstname()." ".$customer->getLastname();
            $customerEmail = $customer->getEmail();
            $senderInfo = [
                'name' => $customerName,
                'email' => $customerEmail
            ];
            $emailTempVariables['name'] = $sellerName;
            $emailTempVariables['subject'] = $postData['subject'];
            $emailTempVariables['quantity'] = $postData['bulk_order_qty'];
            try {
                $template = \Webkul\Requestforquote\Helper\Data::XML_PATH_EMAIL_TO_SELLER;
                $this->dataHelper->customMailSendMethod(
                    $emailTempVariables,
                    $senderInfo,
                    $receiverInfo,
                    $template
                );
            } catch (\Exception $e) {
                $this->messageManager->addError($e->getMessage());
            }
        }
    }
}
