<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Requestforquote
 * @author    Webkul
 * @copyright Copyright (c) Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\Requestforquote\Helper;

use Magento\Framework\UrlInterface;
use Webkul\Marketplace\Model\ResourceModel\Seller\CollectionFactory as SellerCollection;

/**
 * Custom Requestforquote Data helper.
 */
class Data extends \Magento\Framework\App\Helper\AbstractHelper
{
    const XML_PATH_EMAIL_TO_SELLER = 'requestforquote/requestforquote_mailTemplate/rfq_mailtToSellerTemplate';

    const XML_PATH_REPLY_MAIL_TO_SELLER = 'requestforquote/requestforquote_mailTemplate/rfq_replyMailToSellerTemplate';

    const XML_PATH_REPLY_MAIL_TO_CUSTOMER = 'requestforquote/requestforquote_mailTemplate/rfq_replyMailToCustomer';

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Magento\Directory\Model\Currency
     */
    private $currency;

    /**
     * @var \Magento\Framework\Translate\Inline\StateInterface
     */
    private $inlineTranslation;

    /**
     * @var \Magento\Framework\Mail\Template\TransportBuilder
     */
    private $transportBuilder;

    /**
     * @var \Magento\Checkout\Model\Session
     */
    private $checkoutSession;

    /**
     * @var \Magento\Directory\Model\CountryFactory
     */
    private $countryFactory;

    /**
     * @var \Magento\Framework\App\Cache\TypeListInterface
     */
    private $cacheTypeList;

    /**
     * @var \Magento\Framework\App\Cache\Frontend\Pool
     */
    private $cacheFrontendPool;

    /**
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    private $timezoneInterface;

    /**
     * @var \Magento\Framework\Locale\CurrencyInterface
     */
    private $localeCurrency;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    private $jsonHelper;

    /**
     * @var Magento\Directory\Helper\Data
     */
    protected $magentoDirectoryHelper;

    /**
     * @var SellerCollection
     */
    private $sellerCollectionFactory;

    /**
     * __construct function
     *
     * @param \Magento\Framework\App\Helper\Context                            $context
     * @param \Magento\Store\Model\StoreManagerInterface                       $storeManager
     * @param \Magento\Directory\Model\Currency                                $currency
     * @param \Magento\Framework\Translate\Inline\StateInterface               $inlineTranslation
     * @param \Magento\Framework\Mail\Template\TransportBuilder                $transportBuilder
     * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface             $timezoneInterface
     * @param \Magento\Checkout\Model\Session                                  $checkoutSession
     * @param \Magento\Directory\Model\CountryFactory                          $countryFactory
     * @param \Magento\Framework\App\Cache\TypeListInterface                   $cacheTypeList
     * @param \Magento\Framework\App\Cache\Frontend\Pool                       $cacheFrontendPool
     * @param \Magento\Framework\Json\Helper\Data                              $jsonHelper
     * @param \Webkul\Marketplace\Model\ResourceModel\Seller\CollectionFactory $sellerCollectionFactory
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Directory\Model\Currency $currency,
        \Magento\Framework\Translate\Inline\StateInterface $inlineTranslation,
        \Magento\Framework\Mail\Template\TransportBuilder $transportBuilder,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $timezoneInterface,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Directory\Model\CountryFactory $countryFactory,
        \Magento\Framework\App\Cache\TypeListInterface $cacheTypeList,
        \Magento\Framework\App\Cache\Frontend\Pool $cacheFrontendPool,
        \Magento\Framework\Locale\CurrencyInterface $localeCurrency,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Magento\Directory\Helper\Data $helperDirectory,
        SellerCollection $sellerCollectionFactory
    ) {
        $this->storeManager = $storeManager;
        $this->currency = $currency;
        $this->scopeConfig = $context;
        $this->inlineTranslation = $inlineTranslation;
        $this->transportBuilder = $transportBuilder;
        $this->timezoneInterface = $timezoneInterface;
        $this->checkoutSession = $checkoutSession;
        $this->countryFactory = $countryFactory;
        parent::__construct($context);
        $this->cacheTypeList = $cacheTypeList;
        $this->cacheFrontendPool = $cacheFrontendPool;
        $this->localeCurrency = $localeCurrency;
        $this->jsonHelper = $jsonHelper;
        $this->magentoDirectoryHelper = $helperDirectory;
        $this->sellerCollectionFactory = $sellerCollectionFactory;
    }

    /**
     * get minimum quantity for quote
     *
     * @return int
     */
    public function getMinQtyForRfq()
    {
        $quantity =  $this->scopeConfig
            ->getValue(
                'requestforquote/requestforquote_settings/requestforquote_minimumQuantity',
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        return $quantity;
    }

    /**
     * check admin approval required
     *
     * @return int
     */
    public function getApprovalRequiredStatus()
    {
        $status =  $this->scopeConfig
            ->getValue(
                'requestforquote/requestforquote_settings/requestforquote_quoteApproval',
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        return $status;
    }

    /**
     * get the status of the rfq module
     *
     * @return int
     */
    public function getStatusOfModuleFromConfig()
    {
        $status =  $this->scopeConfig
            ->getValue(
                'requestforquote/requestforquote_settings/requestforquote_account',
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        return $status;
    }

    /**
     * get allowed file extentions
     *
     * @return string
     */
    public function getAllowedExt()
    {
        $quantity =  $this->scopeConfig
            ->getValue(
                'requestforquote/requestforquote_settings/requestforquote_allowed_extension',
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE
            );
        return $quantity;
    }

    /**
     * Return store
     *
     * @return Store
     */
    public function getStore()
    {
        return $this->storeManager->getStore();
    }

    /**
     * Return store configuration value of your template field that which id you set for template
     *
     * @param  string $path
     * @param  int    $storeId
     * @return mixed
     */
    protected function getConfigValue($path, $storeId)
    {
        return $this->scopeConfig->getValue(
            $path,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Return template id according to store
     *
     * @return mixed
     */
    public function getTemplateId($xmlPath)
    {
        return $this->getConfigValue($xmlPath, $this->getStore()->getStoreId());
    }

    /**
     * [generateTemplate description]  with template file and tempaltes variables values
     *
     * @param  Mixed $emailTemplateVariables
     * @param  Mixed $senderInfo
     * @param  Mixed $receiverInfo
     * @return void
     */
    public function generateTemplate($emailTemplateVariables, $senderInfo, $receiverInfo)
    {
        $template =  $this->transportBuilder->setTemplateIdentifier($this->temp_id)
            ->setTemplateOptions(
                [
                        /* here you can defile area and store of template for which you prepare it */
                        'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                        'store' => $this->storeManager->getStore()->getId(),
                    ]
            )
            ->setTemplateVars($emailTemplateVariables)
            ->setFrom($senderInfo)
            ->addTo($receiverInfo['email'], $receiverInfo['name']);
        return $this;
    }

    /**
     * [customMailSendMethod description]
     *
     * @param  Mixed $emailTemplateVariables
     * @param  Mixed $senderInfo
     * @param  Mixed $receiverInfo
     * @return void
     */
    public function customMailSendMethod($emailTemplateVariables, $senderInfo, $receiverInfo, $template)
    {
        $this->temp_id = $this->getTemplateId($template);
        $this->inlineTranslation->suspend();
        $this->generateTemplate($emailTemplateVariables, $senderInfo, $receiverInfo);
        $transport = $this->transportBuilder->getTransport();
        $transport->sendMessage();
        $this->inlineTranslation->resume();
    }

    /**
     * get converted time according to locale
     *
     * @param  timestmap $dateTime
     * @return timestamp
     */
    public function getDateTimeAccordingTimeZone($dateTime)
    {
        $dateTimeAsTimeZone = $this->timezoneInterface->date(date($dateTime))
            ->format('d-m-Y h:i A');
        return $dateTimeAsTimeZone;
    }

    /**
     * get the checkout session object
     *
     * @return \Magento\Checkout\Model\Session
     */
    public function getCheckoutSession()
    {
        return $this->checkoutSession;
    }

    /**
     * get country name from crountry code
     *
     * @param  string $countryCode
     * @return string
     */
    public function getCountryname($countryCode)
    {
        $country = $this->countryFactory->create()->loadByCode($countryCode);
        return $country->getName();
    }

    /**
     * clearCache function
     *
     * @return void
     */
    public function clearCache()
    {
        $types = [
            'config',
            'layout',
            'block_html',
            'collections',
            'reflection',
            'db_ddl',
            'eav',
            'config_integration',
            'config_integration_api',
            'full_page',
            'translate',
            'config_webservice'
        ];
        foreach ($types as $type) {
            $this->cacheTypeList->cleanType($type);
        }
        foreach ($this->cacheFrontendPool as $cacheFrontend) {
            $cacheFrontend->getBackend()->clean();
        }
    }

    /**
     * Encodes the given $arr array which is
     * encoded in the array format
     *
     * @param  string $string
     * @return array
     */
    public function jsonEncoder($arr = [])
    {
        $encodedData = $this->jsonHelper->jsonEncode($arr);
        return $encodedData;
    }

    /**
     * decodes the given $arr array which is
     * encoded in the array format
     *
     * @param string $json
     * @return array
     */
    public function jsonDecoder($json)
    {
        $encodedData = $this->jsonHelper->jsonDecode($json);
        return $encodedData;
    }

    /**
     * validatePostData function
     *
     * @param  array $data
     * @return array
     */
    public function validatePostData($data = [])
    {
        if (!isset($data['subject']) || $data['subject'] == '') {
            return ['status' => false, 'error_msg' => __('subject is required field')];
        }
        if (!isset($data['bulk_order_qty']) || $data['bulk_order_qty'] == '') {
            return ['status' => false, 'error_msg' => __('subject is required field')];
        }
        if ($data['bulk_order_qty'] <= 0) {
            return ['status' => false, 'error_msg' => __('Please enter bulk order qty greater than 0')];
        }
        if ($data['bulk_order_qty'] < $data['bulk_order_qty_hidden']) {
            return [
                'status' => false,
                'error_msg' => __(
                    'Please enter bulk order qty greater than or equal to'.$data['bulk_order_qty_hidden']
                )
            ];
        }
        $return = ['status' => true];
        if (isset($data['atr'])) {
            foreach ($data['atr'] as $key => $val) {
                if ($data[$key.'_hidden'] == 'required' && $val == '') {
                    $attrName = explode('-', $key);
                    $return = ['status' => false, 'error_msg' => __($attrName.' is required field.')];
                    break;
                }
            }
        }
        return $return;
    }

    /**
     * get current currency symbol
     *
     * @return string
     */
    public function getCurrentCurrencySymbol()
    {
        $symbol = $this->localeCurrency->getCurrency(
            $this->getCurrentCurrencyCode()
        )->getSymbol();
        if ($symbol == '' || $symbol == null) {
            return $this->getCurrentCurrencyCode();
        }
        return $symbol;
    }

    // return currency currency code
    public function getCurrentCurrencyCode()
    {
        return $this->storeManager->getStore()->getCurrentCurrencyCode();
    }

    // get base currency code
    public function getBaseCurrencyCode()
    {
        return $this->storeManager->getStore()->getBaseCurrencyCode();
    }

    // convert currency amount
    public function getwkconvertCurrency($fromCurrency, $toCurrency, $amount)
    {
        $currentCurrencyCode = $this->getCurrentCurrencyCode();
        $baseCurrencyCode = $this->getBaseCurrencyCode();
        $allowedCurrencies = $this->getConfigAllowCurrencies();
        $rates = $this->getCurrencyRates(
            $baseCurrencyCode,
            array_values($allowedCurrencies)
        );
        if (empty($rates[$fromCurrency])) {
            $rates[$fromCurrency] = 1;
        }

        if ($baseCurrencyCode == $toCurrency) {
            $currencyAmount = $amount/$rates[$fromCurrency];
        } else {
            $amount = $amount/$rates[$fromCurrency];
            $currencyAmount = $this->convertCurrency($amount, $baseCurrencyCode, $toCurrency);
        }
        return $currencyAmount;
    }

    // get all allowed currency in system config
    public function getConfigAllowCurrencies()
    {
        return $this->currency->getConfigAllowCurrencies();
    }

    /**
     * Retrieve currency rates to other currencies.
     *
     * @param string     $currency
     * @param array|null $toCurrencies
     *
     * @return array
     */
    public function getCurrencyRates($currency, $toCurrencies = null)
    {
        // give the currency rate
        return $this->currency->getCurrencyRates($currency, $toCurrencies);
    }

    /**
     * convert amount according to currenct currency
     */
    public function convertCurrency($amount, $from, $to)
    {
        $finalAmount = $this->magentoDirectoryHelper
            ->currencyConvert($amount, $from, $to);

        return $finalAmount;
    }

    /**
     * checkForSellerExistence function
     *
     * @return void
     */
    public function checkForSellerExistence()
    {
        $collection = $this->sellerCollectionFactory->create();
        $collection->addFieldToFilter('is_seller', 1);
        return $collection->getSize();
    }
}
